// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import SwiftUI
import MSSImageScanner
import CoreMedia

struct MainView: View {
    // Title labels
    private let titleAppName      = "Image Scanner SDK Sample"
    private let titleImageContent = "Image content:"
    private let titleImageType    = "Image type:"
    
    // Button titles
    private let buttonTitleSystemScanner  = "Launch System Camera"
    private let buttonTitleUIKitScanner = "Launch UIKit Scanner"
    private let buttonTitleSwiftUIScanner = "Launch SwiftUI Scanner"
        
    @StateObject private var mainViewModel = MainViewModel()
    
    var body: some View {
        VStack(spacing: 24) {
            
            VStack(spacing: 0) {
                Image("onespan_logo")
                    .resizable()
                    .scaledToFit()
                    .frame(height: 80)
                    .padding(.top, 20)
                
                Text(titleAppName)
                    .font(.headline)
            }
            
            // Image Type Section
            VStack(alignment: .leading, spacing: 8) {
                Text(titleImageType)
                    .font(.subheadline)
                    .foregroundColor(.secondary)
                
                Text(mainViewModel.scanResultType)
                    .font(.body)
                    .padding()
                    .frame(maxWidth: .infinity, alignment: .leading)
                    .background(Color(.systemGray6))
                    .cornerRadius(8)
            }
            .padding(.horizontal)
            
            VStack(alignment: .leading, spacing: 8) {
                Text(titleImageContent)
                    .font(.subheadline)
                    .foregroundColor(.secondary)
                
                ScrollView(.vertical) {
                    Text(mainViewModel.scanResultLabel)
                        .font(.system(.body, design: .monospaced))
                        .padding()
                        .frame(maxWidth: .infinity, alignment: .leading)
                        .background(Color(.systemGray6))
                        .cornerRadius(8)
                }
                .frame(height: 150)
            }
            .padding(.horizontal)
            
            Spacer()
            
            // Buttons
            VStack(spacing: 12) {
                Button(buttonTitleSystemScanner) {
                    mainViewModel.isShowingUIImagePickerSample = true
                }
                .buttonStyle(MainActionButtonStyle())
                
                Button(buttonTitleUIKitScanner) {
                    mainViewModel.isShowingUIKitSample = true
                }
                .buttonStyle(MainActionButtonStyle())
                
                Button(buttonTitleSwiftUIScanner) {
                    mainViewModel.isShowingSwiftUISample = true
                }
                .buttonStyle(MainActionButtonStyle())
            }
            .padding(.horizontal)
            
            Spacer()
        }
        .padding(.bottom)
        .fullScreenCover(isPresented: $mainViewModel.isShowingUIImagePickerSample) {
            UIImagePickerSampleView(isPresented: $mainViewModel.isShowingUIImagePickerSample,
                                    scanResult: $mainViewModel.scanResult)
            .modifier(FullScreenCoverCustom())
        }
        .fullScreenCover(isPresented: $mainViewModel.isShowingUIKitSample) {
            UIKitSampleView(isPresented: $mainViewModel.isShowingUIKitSample,
                            scanResult: $mainViewModel.scanResult)
            .edgesIgnoringSafeArea(.top)
            .modifier(FullScreenCoverCustom())
        }
        .fullScreenCover(isPresented: $mainViewModel.isShowingSwiftUISample) {
            SwiftUISampleView(isPresented: $mainViewModel.isShowingSwiftUISample,
                              scanResult: $mainViewModel.scanResult,
                              viewModel: .init())
            .modifier(FullScreenCoverCustom())
        }
        .transaction({ transaction in
            transaction.disablesAnimations = true
        })
        .onAppear {
            print("Version: \(QRCodeScannerSDK.version)")
        }
    }
}

struct MainActionButtonStyle: ButtonStyle {
    func makeBody(configuration: Configuration) -> some View {
        configuration.label
            .font(.body)
            .frame(maxWidth: .infinity)
            .padding()
            .background(Color("primary"))
            .foregroundColor(Color.black.opacity(0.85))
            .cornerRadius(10)
    }
}

struct FullScreenCoverCustom: ViewModifier {
    func body(content: Content) -> some View {
        ZStack {
            Color.black
                .edgesIgnoringSafeArea(.all)
            
            content
                .frame(maxWidth: .infinity, maxHeight: .infinity)
                .statusBarHidden(false)
        }
    }
}

struct MainView_Previews: PreviewProvider {
    static var previews: some View {
        MainView()
    }
}
